/* General type definitions, and procedure declarations. */
#include "lktMain.h"

/* Program version specification. */
#define program_version "0.3"

static char *program_path = "locktones"; // The full path to the executing binary.
static char *program_name = NULL; // The name (last path component) of the executing binary.

const char *character_options = "+hvc:d:i:n:p:s:"; // Program invocation option definitions.

static int poll_interval = 100000; // Micro-seconds between each poll of the lock states.
static int tone_duration = 200000; // Duration of each tone (in micro-seconds).
static int caps_pitch = 300; // The current caps lock pitch.
static int insert_pitch = 450; // The current insert lock pitch.
static int num_pitch = 600; // The current num lock pitch.
static int scroll_pitch = 1200; // The current scroll lock pitch.

static int lock_flags = 0; // The current states of the keyboard locks.
typedef struct { // The definition of an entry in the lock table.
   int flag; // The flag bit for this lock.
   int *pitch; // The pitch for this lock.
} lock_entry;
static const lock_entry lock_table[] = { // The table of all of the supported locks.
   {CAPS_FLAG  , &caps_pitch},
   {INSERT_FLAG, &insert_pitch},
   {NUM_FLAG   , &num_pitch},
   {SCROLL_FLAG, &scroll_pitch}
};
static const int lock_count = sizeof(lock_table) / sizeof(lock_entry); // The number of locks in the table.
static int lock_index = 0; // The next lock to check.

/* Terminate the program with the conventional exit status used for syntax errors. */
void syntax_error () {
   exit(2);
}

/* Write a message to standard error describing the most recent system error (value of errno)
 * and then terminate the program with a non-zero exit status.
 */
void system_error (const char *action) {
   int error_number = errno;
   char message_buffer[0X100];
   sprintf(message_buffer, "%s error %d", action, error_number);
   if ((error_number > 0) && (error_number <= sys_nerr)) {
      /* The system can supply a textual explanation of this error. */
      strcat(message_buffer, ": ");
      strcat(message_buffer, sys_errlist[error_number]);
   }
   strcat(message_buffer, ".");
   fprintf(stderr, "%s: %s\n", program_path, message_buffer);
   exit(3);
}

/* Determine the path to, and name of, the executing binary. */
static void identify_program (int argc, char **argv) {
   if (argv) {
      /* An argument vector has been supplied. */
      if (*argv) {
	 /* The path to the executing binary has been supplied. */
	 program_path = *argv;
      }
   }
   if ((program_name = strrchr(program_path, '/'))) {
      /* The path contains multiple components. */
      ++program_name;
   } else {
      /* The path contains just one component. */
      program_name = program_path;
   }
}

/* Process the positional arguments with which the program was invoked.
 * At present, no positional arguments are supported.
 */
static void process_parameters (int argc, char **argv) {
   if (argc > 0) {
      /* There remains at least one unprocessed positional argument. */
      fprintf(stderr, "%s: too many parameters.\n", program_path);
      syntax_error();
   }
}

/* Interpret the operand of the option currently being processed as an integer,
 * and insure that its value is within an acceptable range.
 * If it begins with either "0x" or "0X", then it's interpreted as a hexadecimal number;
 * if it begins with "0" then it's interpreted as an octal number;
 * if it begins with "1" through "9", then it's interpreted as a decimal number.
 */
static int integer_operand (const char *description, const int *minimum, const int *maximum) {
   const char *string = get_operand();
   if (*string) {
      /* The operand contains at least one character. */
      char *end;
      int integer = strtol(string, &end, 0);
      if (*end == 0) {
	 /* No non-numeric character was encountered. */
	 int acceptable = TRUE;
	 if (minimum != NULL) {
	    /* A minimum value has been specified. */
	    if (integer < *minimum) {
	       /* The value is less than the specified minimum value. */
	       acceptable = FALSE;
	    }
	 }
	 if (maximum != NULL) {
	    /* A maximum value has been specified. */
	    if (integer > *maximum) {
	       /* The value is greater than the specified maximum value. */
	       acceptable = FALSE;
	    }
	 }
         if (acceptable) {
	    /* The value is within an acceptable range. */
	    return integer;
	 }
      }
   }
   fprintf(stderr, "%s: invalid %s -- %s\n", program_path, description, string);
   syntax_error();
}

/* Interpret the operand of the option currently being processed as a time duration in tenths of seconds. */
static int time_operand (const char *description) {
   static int minimum = 1;
   static int maximum = 10;
   return integer_operand(description, &minimum, &maximum) * 100000;
}

/* Interpret the operand of the option currently being processed as a pitch in Herz. */
static int pitch_operand (const char *description) {
   static int minimum = 30;
   static int maximum = 10000;
   return integer_operand(description, &minimum, &maximum);
}

/* Process the optional arguments with which the program was invoked.
 * When done, adjust argc and argv so that they only reference the positional arguments.
 */
static void process_options (int *argcp, char ***argvp) {
   int current_option;
   int exit_immediately = FALSE;
   int display_usage = FALSE;
   int display_version = FALSE;
   while ((current_option = get_option(*argcp, *argvp)) != EOF) {
      switch (current_option) {
	 default: // A new option has been added to the table without adding a corresponding case clause below.
	    fprintf(stderr, "%s: option not implemented -- %c\n", program_path, current_option);
	 case '?': // An unknown option has been specified (error already displayed).
	 case ':': // A required operand has not been supplied (error already displayed).
	    syntax_error();
         case 'h': // --help: Write usage information to standard output, and then exit.
	    display_usage = TRUE;
	    break;
         case 'v': // --version: Write version information to standard output, and then exit.
	    display_version = TRUE;
	    break;
         case 'c': // --caps: Specify the pitch of the caps lock tone.
	    caps_pitch = pitch_operand("caps lock pitch");
	    break;
         case 'd': // --duration: Specify the duration of each tone.
	    tone_duration = time_operand("tone duration");
	    break;
         case 'i': // --insert: Specify the pitch of the caps lock tone.
	    insert_pitch = pitch_operand("caps lock pitch");
	    break;
         case 'n': // --num: Specify the pitch of the num lock tone.
	    num_pitch = pitch_operand("num lock pitch");
	    break;
         case 'p': // --poll: Specify the interval at which the lock states are checked.
	    poll_interval = time_operand("lock state poll interval");
	    break;
         case 's': // --scroll: Specify the pitch of the scroll lock tone.
	    scroll_pitch = pitch_operand("scroll lock pitch");
	    break;
      }
   }
   *argcp -= optind; // The positional argument count.
   *argvp += optind; // The address of the first positional argument.
   if (display_version) {
      /* The "version" option has been specified. */
      printf("%s %s\n", program_name, program_version);
      exit_immediately = 1;
   }
   if (display_usage) {
      /* The "help" option has been specified. */
      const static char *usage_lines[] = {
	 "",
	 "Use the PC speaker to audibly indicate the state of the keyboard locks.",
	 "",
	 "{-p | --poll} time           keyboard poll interval",
	 "{-d | --duration} time       duration of each tone",
	 "{-c | --caps} frequency      frequency of caps lock tone",
	 "{-i | --insert} frequency    frequency of insert lock tone",
	 "{-n | --num} frequency       frequency of num lock tone",
	 "{-s | --scroll} frequency    frequency of scroll lock tone",
	 "{-h | --help}                write usage information to standard output",
	 "{-v | --version}             write version information to standard output",
	 "",
	 "A \"time\" operand is an integer from 1 through 10 (tenths of a second).",
	 "A \"tone\" operand is an integer from 30 through 10000 (Herz).",
         NULL
      };
      const char **usage_line = usage_lines;
      printf("Usage: %s [-option value] ... \n", program_name);
      while (*usage_line) {
         printf("%s\n", *usage_line++);
      }
      exit_immediately = 1;
   }
   if (exit_immediately) {
      /* At least one of the "information only" options has been specified. */
      exit(0);
   }
}

/* Start the next tone. */
void next_tone () {
   int loop_limit;
   for (loop_limit=lock_count; loop_limit>0; --loop_limit) {
      const lock_entry *lock = &lock_table[lock_index];
      int locked = (lock_flags & lock->flag) != 0;
      lock_index = (lock_index + 1) % lock_count;
      if (locked) {
	 start_tone(*lock->pitch);
	 return;
      }
   }
   stop_timer();
   stop_tone();
}

/* Check the lock states, and take appropriate action if they have changed. */
void check_flags (int flags) {
   if (flags != lock_flags) {
      if (lock_flags = flags) {
	 /* At least one lock is on. */
	 start_timer(tone_duration);
      } else {
	 /* No locks are on. */
	 stop_timer();
	 stop_tone();
      }
   }
}

/* The main program. */
int main (int argc, char **argv) {
   identify_program(argc, argv);
   process_options(&argc, &argv);
   process_parameters(argc, argv);
   monitor_locks(poll_interval);
}
