/* General includes for all Unix systems. */
#include <stdlib.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/time.h>
#include <fcntl.h>
#include <errno.h>
#include <signal.h>
#include <string.h>
#include <ctype.h>
#include <getopt.h>
#include <stdio.h>

/* Linux-specific includes and definitions. */
#include <linux/kd.h>
#define CAPS_FLAG LED_CAP
#define NUM_FLAG LED_NUM
#define SCROLL_FLAG LED_SCR
#define CONSOLE_PATH "/dev/tty0"
#define TONE_CLOCK 1190000
#define tone_value(frequency) (TONE_CLOCK / (frequency))

/* Program version specification. */
#define program_version "0.2"

/* Boolean constants. */
#define TRUE (0 == 0)
#define FALSE (!TRUE)

static char *program_path = "locktones"; // The full path to the executing binary.
static char *program_name = NULL; // The name (last path component) of the executing binary.

static int become_daemon = FALSE; // Run in the foreground (FALSE), or background (TRUE).
static int poll_interval = 100000; // Micro-seconds between each poll of the lock states.
static int tone_duration = 200000; // Duration of each tone (in micro-seconds).
static int caps_tone = tone_value(297); // The current caps lock tone.
static int num_tone = tone_value(595); // The current num lock tone.
static int scroll_tone = tone_value(1190); // The current scroll lock tone.

static int console_descriptor = -1; // A handle to the console I/O subsystem.
static int lock_flags = 0; // The current states of the keyboard locks.
typedef struct { // The definition of an entry in the lock table.
   int flag; // The flag bit for this lock.
   int *tone; // The tone for this lock.
} lock_entry;
static const lock_entry lock_table[] = { // The table of all of the supported locks.
   {CAPS_FLAG  , &caps_tone},
   {NUM_FLAG   , &num_tone},
   {SCROLL_FLAG, &scroll_tone}
};
static const int lock_count = sizeof(lock_table) / sizeof(lock_entry); // The number of locks in the table.
static int lock_index = 0; // The next lock to check.

/* Terminate the program with the conventional exit status used for syntax errors. */
static void syntax_error () {
   exit(2);
}

/* Write a message to standard error describing the most recent system error (value of errno)
 * and then terminate the program with a non-zero exit status.
 */
static void system_error (const char *action) {
   int error_number = errno;
   char message_buffer[0X100];
   sprintf(message_buffer, "%s error %d", action, error_number);
   if ((error_number > 0) && (error_number <= sys_nerr)) {
      /* The system can supply a textual explanation of this error. */
      strcat(message_buffer, ": ");
      strcat(message_buffer, sys_errlist[error_number]);
   }
   strcat(message_buffer, ".");
   fprintf(stderr, "%s: %s\n", program_path, message_buffer);
   exit(3);
}

/* Determine the path to, and name of, the executing binary. */
static void identify_program (int argc, char **argv) {
   if (argv) {
      /* An argument vector has been supplied. */
      if (*argv) {
	 /* The path to the executing binary has been supplied. */
	 program_path = *argv;
      }
   }
   if ((program_name = strrchr(program_path, '/'))) {
      /* The path contains multiple components. */
      ++program_name;
   } else {
      /* The path contains just one component. */
      program_name = program_path;
   }
}

/* Process the positional arguments with which the program was invoked.
 * At present, no positional arguments are supported.
 */
static void process_parameters (int argc, char **argv) {
   if (argc > 0) {
      /* There remains at least one unprocessed positional argument. */
      fprintf(stderr, "%s: too many parameters.\n", program_path);
      syntax_error();
   }
}

/* Return a string containing the operand for the option being currently processed. */
static const char *string_operand () {
   return optarg;
}

/* Interpret the operand of the option currently being processed as an integer,
 * and insure that its value is within an acceptable range.
 * If it begins with either "0x" or "0X", then it's interpreted as a hexadecimal number;
 * if it begins with "0" then it's interpreted as an octal number;
 * if it begins with "1" through "9", then it's interpreted as a decimal number.
 */
static int integer_operand (const char *description, const int *minimum, const int *maximum) {
   const char *string = string_operand();
   if (*string) {
      /* The operand contains at least one character. */
      char *end;
      int integer = strtol(string, &end, 0);
      if (*end == 0) {
	 /* No non-numeric character was encountered. */
	 int acceptable = TRUE;
	 if (minimum != NULL) {
	    /* A minimum value has been specified. */
	    if (integer < *minimum) {
	       /* The value is less than the specified minimum value. */
	       acceptable = FALSE;
	    }
	 }
	 if (maximum != NULL) {
	    /* A maximum value has been specified. */
	    if (integer > *maximum) {
	       /* The value is greater than the specified maximum value. */
	       acceptable = FALSE;
	    }
	 }
         if (acceptable) {
	    /* The value is within an acceptable range. */
	    return integer;
	 }
      }
   }
   fprintf(stderr, "%s: invalid %s -- %s\n", program_path, description, string);
   syntax_error();
}

/* Interpret the operand of the option currently being processed as a time duration in tenths of seconds. */
static int time_operand (const char *description) {
   static int minimum = 1;
   static int maximum = 10;
   return integer_operand(description, &minimum, &maximum) * 100000;
}

/* Interpret the operand of the option currently being processed as a pitch in Herz. */
static int tone_operand (const char *description) {
   static int minimum = 30;
   static int maximum = 10000;
   return tone_value(integer_operand(description, &minimum, &maximum));
}

/* Process the optional arguments with which the program was invoked.
 * When done, adjust argc and argv so that they only reference the positional arguments.
 */
static void process_options (int *argcp, char ***argvp) {
   static const char *character_options = "+hvbc:d:n:p:s:";
   static const struct option word_options[] = {
      {"help"      , no_argument      , NULL, 'h'},
      {"version"   , no_argument      , NULL, 'v'},
      {"background", no_argument      , NULL, 'b'},
      {"caps"      , required_argument, NULL, 'c'},
      {"duration"  , required_argument, NULL, 'd'},
      {"num"       , required_argument, NULL, 'n'},
      {"poll"      , required_argument, NULL, 'p'},
      {"scroll"    , required_argument, NULL, 's'},
      {NULL        , 0                , NULL, 0  }
   };
   int current_option;
   int exit_immediately = FALSE;
   int display_usage = FALSE;
   int display_version = FALSE;
   while ((current_option = getopt_long(*argcp, *argvp, character_options, word_options, NULL)) != EOF) {
      switch (current_option) {
	 default:
	    fprintf(stderr, "%s: option not implemented -- %c\n", program_path, current_option);
	    syntax_error();
	 case '?':
	 case ':':
	    syntax_error();
         case 'h':
	    display_usage = TRUE;
	    break;
         case 'v':
	    display_version = TRUE;
	    break;
         case 'b':
	    become_daemon = TRUE;
	    break;
         case 'c':
	    caps_tone = tone_operand("caps lock pitch");
	    break;
         case 'd':
	    tone_duration = time_operand("tone duration");
	    break;
         case 'n':
	    num_tone = tone_operand("num lock pitch");
	    break;
         case 'p':
	    poll_interval = time_operand("lock state poll interval");
	    break;
         case 's':
	    scroll_tone = tone_operand("scroll lock pitch");
	    break;
      }
   }
   *argcp -= optind; // The positional argument count.
   *argvp += optind; // The address of the first positional argument.
   if (display_version) {
      /* The "version" option has been specified. */
      printf("%s %s\n", program_name, program_version);
      exit_immediately = 1;
   }
   if (display_usage) {
      /* The "help" option has been specified. */
      const static char *usage_lines[] = {
	 "",
	 "Use the PC speaker to audibly indicate the state of the keyboard locks.",
	 "",
	 "{-b | --background}          execute in the background",
	 "{-p | --poll} time           keyboard poll interval",
	 "{-d | --duration} time       duration of each tone",
	 "{-c | --caps} frequency      frequency of caps lock tone",
	 "{-n | --num} frequency       frequency of num lock tone",
	 "{-s | --scroll} frequency    frequency of scroll lock tone",
	 "{-h | --help}                write usage information to standard output",
	 "{-v | --version}             write version information to standard output",
	 "",
	 "A \"time\" operand is an integer from 1 through 10 (tenths of a second).",
	 "A \"tone\" operand is an integer from 30 through 10000 (Herz).",
         NULL
      };
      const char **usage_line = usage_lines;
      printf("Usage: %s [-option value] ... \n", program_name);
      while (*usage_line) {
         printf("%s\n", *usage_line++);
      }
      exit_immediately = 1;
   }
   if (exit_immediately) {
      /* At least one of the "information only" options has been specified. */
      exit(0);
   }
}

/* Get a handle to the console I/O subsystem. */
static void open_console () {
   if ((console_descriptor = open(CONSOLE_PATH, O_WRONLY)) == -1) {
      /* The console device could not be opened. */
      system_error("console open");
   }
}

/* Fill in the fields of a "timeval" data structure.
 * Assume that all slack bits have previously been cleared.
 */
void set_timeval (struct timeval *tv, int micro_seconds) {
   tv->tv_sec = micro_seconds / 1000000;
   tv->tv_usec = micro_seconds % 1000000;
}

/* Delay program execution for the specified period of time. */
static void delay_execution (int micro_seconds) {
   struct timeval tv;
   memset(&tv, 0, sizeof(tv));
   set_timeval(&tv, micro_seconds);
   select(0, NULL, NULL, NULL, &tv);
}

/* Start the specified tone. */
static void start_tone (int tone) {
   ioctl(console_descriptor, KIOCSOUND, tone);
}

/* Stop the current tone. */
static void stop_tone () {
   ioctl(console_descriptor, KIOCSOUND, 0);
}

/* Stop periodic alarm signal generation. */
static void stop_timer () {
   struct itimerval itv;
   memset(&itv, 0, sizeof(itv));
   set_timeval(&itv.it_value, 0);
   set_timeval(&itv.it_interval, 0);
   setitimer(ITIMER_REAL, &itv, NULL);
}

/* Alarm signal handler to start the next tone. */
static void next_tone (int signal) {
   int loop_limit;
   for (loop_limit=lock_count; loop_limit>0; --loop_limit) {
      const lock_entry *lock = &lock_table[lock_index];
      int locked = (lock_flags & lock->flag) != 0;
      lock_index = (lock_index + 1) % lock_count;
      if (locked) {
	 start_tone(*lock->tone);
	 return;
      }
   }
   stop_timer();
   stop_tone();
}

/* Install the alarm signal handler so that active lock tones can be sounded cyclically. */
static void monitor_timer () {
   struct sigaction sa;
   memset(&sa, 0, sizeof(sa));
   sa.sa_handler = next_tone;
   sigaction(SIGALRM, &sa, NULL);
}

/* Start periodic alarm signal generation for cyclical tone playing. */
static void start_timer () {
   struct itimerval itv;
   memset(&itv, 0, sizeof(itv));
   set_timeval(&itv.it_value, 1);
   set_timeval(&itv.it_interval, tone_duration);
   setitimer(ITIMER_REAL, &itv, NULL);
}

/* Return the current lock states. */
static int get_flags () {
   int flags;
   ioctl(console_descriptor, KDGETLED, &flags);
   return flags;
}

/* Poll the lock states, and take appropriate action if they have changed. */
static void poll_flags () {
   int flags = get_flags();
   if (flags != lock_flags) {
      if (lock_flags = flags) {
	 /* At least one lock is on. */
	 start_timer();
      } else {
	 /* No locks are on. */
	 stop_timer();
      }
   }
}

/* The main program. */
void main (int argc, char **argv) {
   identify_program(argc, argv);
   process_options(&argc, &argv);
   process_parameters(argc, argv);
   open_console();;
   if (become_daemon) {
      /* Execute in the background. */
      pid_t pid = fork();
      if (pid != 0) {
	 if (pid == -1) {
	    system_error("process creation");
	 }
         exit(0);
      }
      fclose(stdin);
      fclose(stdout);
      fclose(stderr);
      setsid();
   }
   monitor_timer();

   /* Loop forever. */
   while (1) {
      poll_flags();
      delay_execution(poll_interval);
   }
}
