#include "lktMain.h"
#include "lktUnix.h"

const struct option word_options[] = {
   {"help"      , no_argument      , NULL, 'h'},
   {"version"   , no_argument      , NULL, 'v'},
   {"active"    , no_argument      , NULL, 'a'},
   {"caps"      , required_argument, NULL, 'c'},
   {"duration"  , required_argument, NULL, 'd'},
   {"insert"    , required_argument, NULL, 'i'},
   {"num"       , required_argument, NULL, 'n'},
   {"poll"      , required_argument, NULL, 'p'},
   {"scroll"    , required_argument, NULL, 's'},
   {"toggle"    , no_argument      , NULL, 't'},
   {NULL        , 0                , NULL, 0  }
};

static int console_descriptor = -1; // A handle to the console I/O subsystem.

/* Get a handle to the console I/O subsystem. */
static void open_console () {
   if ((console_descriptor = open("/dev/tty0", O_WRONLY)) == -1) {
      /* The console device could not be opened. */
      system_error("console open");
   }
}

/* Install a signal handler. */
static void install_handler (int signal, void (*handler)(int signal)) {
   struct sigaction sa;
   clear_variable(sa);
   sa.sa_handler = handler;
   sigaction(signal, &sa, NULL);
}

/* Handle the alarm signal. */
static void alarm_handler (int signal) {
   select_tone();
}

/* Place the program into the background. */
static void become_daemon () {
   fflush(stdout);
   fflush(stderr);
   {
      pid_t pid = fork();
      if (pid != 0) {
	 /* This is the parent process. */
	 if (pid == -1) {
	    /* An error occurred. */
	    system_error("process creation");
	 }
	 printf("%s %s running as process %d.\n", program_path, program_version, pid);
	 exit(0);
      }
   }
   fclose(stdin);
   fclose(stdout);
   fclose(stderr);
   setsid();
}

/* Fill in the fields of a "timeval" data structure.
 * Assume that all slack bits have previously been cleared.
 */
static void set_timeval (struct timeval *tv, int micro_seconds) {
   tv->tv_sec = micro_seconds / 1000000;
   tv->tv_usec = micro_seconds % 1000000;
}

/* Delay program execution for the specified period of time. */
static void delay_execution (int micro_seconds) {
   struct timeval tv;
   clear_variable(tv);
   set_timeval(&tv, micro_seconds);
   select(0, NULL, NULL, NULL, &tv);
}

/* Return the current lock states. */
static int get_flags () {
   int flags = 0;
   int leds = 0;
   if (ioctl(console_descriptor, KDGETLED, &leds) != -1) {
      /* The query of the lock states was successful. */
      if (leds & LED_CAP) {
	 /* The caps lock is active. */
         flags |= caps_flag;
      }
      if (leds & LED_NUM) {
	 /* The num lock is active. */
         flags |= num_flag;
      }
      if (leds & LED_SCR) {
	 /* The scroll lock is active. */
         flags |= scroll_flag;
      }
   }
   return flags;
}

/* Monitor the locks in the background. */
void monitor_locks (int poll_interval) {
   open_console();
   become_daemon();
   install_handler(SIGALRM, alarm_handler);

   /* Loop forever. */
   while (TRUE) {
      check_flags(get_flags());
      delay_execution(poll_interval);
   }
}

/* Schedule an alarm signal. */
static void set_alarm (int interval) {
   struct itimerval itv;
   clear_variable(itv);
   set_timeval(&itv.it_value, interval);
   setitimer(ITIMER_REAL, &itv, NULL);
}

/* Cancel any pending alarm signal. */
static void cancel_alarm () {
   struct itimerval itv;
   clear_variable(itv);
   setitimer(ITIMER_REAL, &itv, NULL);
}

/* Start the specified tone. */
void start_tone (int pitch, int duration) {
   if (ioctl(console_descriptor, KIOCSOUND, (1190000 / pitch)) != -1) {
      /* The tone has been started. */
      set_alarm(duration);
   }
}

/* Stop the current tone. */
void stop_tone () {
   cancel_alarm();
   ioctl(console_descriptor, KIOCSOUND, 0);
}

/* Get the next program invocation option. */
char get_option (int argc, char **argv) {
   return getopt_long(argc, argv, character_options, word_options, NULL);
}

/* Get the operand for the current program invocation option. */
const char *get_operand () {
   return optarg;
}
