#include "lktMain.h"
#include "lktDjgpp.h"

int _stklen = 8192;
int _crt0_startup_flags = _CRT0_FLAG_LOCK_MEMORY;

static volatile int poll_length = 0; // Poll interval in ticks.
static volatile int poll_left = 0; // Ticks left until next poll.

static volatile int tone_length = 0; // Tone duration in ticks.
static volatile int tone_left = 0; // Ticks left for current tone.

/* Return the current lock states. */
static int get_flags () {
   int flags = 0;
   int status = _bios_keybrd(_KEYBRD_SHIFTSTATUS);
   if (status & 0X10) {
      flags |= SCROLL_FLAG;
   }
   if (status & 0X20) {
      flags |= NUM_FLAG;
   }
   if (status & 0X40) {
      flags |= CAPS_FLAG;
   }
   if (status & 0X80) {
      flags |= INSERT_FLAG;
   }
   return flags;
}

/* Handle clock ticks. */
static void tick_handler () {
   if (poll_left) {
      /* The poll interval has not yet expired. */
      --poll_left;
   } else {
      /* The poll interval has expired. */
      poll_left = poll_length - 1;
      check_flags(get_flags());
   }
   if (tone_length) {
      /* The tone timer is active. */
      if (tone_left) {
	 /* The current tone has not yet expired. */
         --tone_left;
      } else {
	 /* The current tone has expired. */
         tone_left = tone_length - 1;
	 next_tone();
      }
   }
}

/* Install an interrupt handler. */
static void install_handler (int interrupt, void (*handler)(void)) {
   _go32_dpmi_seginfo si;
   si.pm_selector = _my_cs();
   si.pm_offset = (unsigned)handler;
   _go32_dpmi_chain_protected_mode_interrupt_vector(interrupt, &si);
}

/* Terminate the program but leave it resident. */
static void stay_resident () {
   __dpmi_regs regs;
   __djgpp_exception_toggle();
   regs.x.ax = 0x3100; // INT21[AX]: TSR=0X31, PgmRetCode=0X00
   regs.x.dx = (256) / 16; // INT21[DX]: Paragraphs (16-byte blocks) to keep.
   __dpmi_int(0x21, &regs);
}

/* Monitor the locks in the background. */
void monitor_locks (int poll_interval) {
   poll_length = tick_count(poll_interval);
   install_handler(8, &tick_handler);
   stay_resident();
}

/* Start the specified tone. */
void start_tone (int pitch) {
   sound(pitch);
}

/* Stop the current tone. */
void stop_tone () {
   nosound();
}

/* Start periodic alarm signal generation for cyclical tone playing. */
void start_timer (int interval) {
   tone_length = tick_count(interval);
}

/* Stop periodic alarm signal generation. */
void stop_timer () {
   tone_length = 0;
}

/* Get the next program invocation option. */
char get_option (int argc, char **argv) {
   return getopt(argc, argv, character_options);
}

/* Get the operand for the current program invocation option. */
const char *get_operand () {
   return optarg;
}
